<?php

/**
 * This is the model class for table "Zlecenie".
 *
 * The followings are the available columns in table 'Zlecenie':
 * @property double $id
 * @property double $idDziedziny
 * @property double $poziom
 * @property string $kategoria
 * @property double $czas
 * @property double $cena
 * @property double $kara
 * @property string $czyPrzetarg
 * @property string $tresc
 * @property string $urlObrazka
 * @property string $jednostka
 * @property string $czyEgzamin
 * @property string $jezyk
 * @property string $wylaczone
 *
 * @property Dziedzina           $dziedzina
 * @property ZlecenieFirmy[]     $zlecenieFirmies
 * @property ZlecenieOdpowiedz[] $odpowiedziZlecenia
 *
 * @property string $poziomStr
 * @property string $poziomNum
 * @property string $czasNum
 * @property string $scoreForOrder
 * @property string $kwotaKary
 * @property string $kwotaKaryCurr
 * @property string $czasTrwaniaStr
 */
class Zlecenie extends CActiveRecord
{
	public $moodle;
	public $image;
	public $tempFolderName;
	public $tempImageName;

	const LEVEL_EASY = 1;
	const LEVEL_HARD = 2;
	const LEVEL_VERY_HARD = 3;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Zlecenie the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Zlecenie';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('*', 'importFilter', 'on' => 'import'),
			array('kategoria, czas, cena, kara', 'safe', 'on'=>'import'),

			array('idDziedziny, poziom, tresc', 'required'),
			array('idDziedziny, poziom, czas, cena, kara', 'numerical'),
			array('kategoria, jednostka', 'length', 'max'=>64),
			array('czyPrzetarg, czyEgzamin', 'length', 'max'=>1),
			array('jezyk', 'length', 'max'=>10),
			array('jezyk', 'validateLanguage'),
			array('czyEgzamin', 'default', 'value'=>1),
			array('kara', 'length', 'max'=>100, 'min'=>0),
			array('tresc, urlObrazka', 'length', 'max'=>2000),
			array('image', 'file','safe' => true, 'types'=>'jpg, gif, png', 'allowEmpty'=>true),
			array('tempFolderName,tempImageName', 'safe'),
			array('moodle, wylaczone', 'safe'),

			array('id, idDziedziny, poziom, kategoria, czas, cena, kara, czyPrzetarg,
				tresc, urlObrazka, czyEgzamin, jezyk, wylaczone', 'safe', 'on'=>'search'),
		);
	}

	private function checkList($propName, $errorMsg, &$list)
	{
		$old = $this->$propName;
		if (!isset($list[$old]))
		{
			$name = strtolower(trim($old));
			$this->$propName = null;
			foreach ($list as $id => $value)
				if (strtolower($value) == $name || $value == ucfirst($name)) {
					$this->$propName = $id;
					return true;
				}
		}
		else return true;
		$this->$propName = -1;
		$this->addError($propName, str_replace('#value#', $old, $errorMsg));
		return false;
	}

	private function getBoolean($fieldname)
	{
		$old = $this->$fieldname;
		$value = strtolower(trim($old));
		if (in_array($value, array('0', 'nie', 'fałsz', 'n', 'no', 'false', 'falsz')))
			$this->$fieldname = 0;
		else if (in_array($value, array('1', 'tak', 'prawda', 't', 'yes', 'true', 'y')))
			$this->$fieldname = 1;
		else {
			$this->$fieldname = 2;
			$this->addError($fieldname, 'Nieprawidłowa wartośc logiczna dla ' . $fieldname . ' (' . $old . ').');
		}
	}

	public function importFilter()
	{
		$dziedziny = Dziedzina::listAll();
		$this->checkList('idDziedziny', 'Dziedzina nie istnieje (#value#).', $dziedziny);

		$this->checkList('poziom', 'Poziom nie istnieje (#value#).', self::$_poziomy);

		if(empty($this->kategoria))
			$this->kategoria = 'Ogólna';

		$this->czas = $this->getCzasNum();
		$this->kara = $this->getKaraNum();

		if(empty($this->cena))
			$this->cena = $this->getParamDefaultValue('CENA', $this->idDziedziny, $this->poziom);

		if(empty($this->czyPrzetarg))
			$this->czyPrzetarg = 0;

		$this->getBoolean('czyPrzetarg');

		if($this->tresc === 'RAND')
			$this->tresc = md5(time() . uniqid());

		$this->getBoolean('czyEgzamin');
	}

	public function validateLanguage($attribute, $params)
	{
		$oldJ = $this->$attribute;
		$this->$attribute = strtolower(trim($this->$attribute));
		if (empty($this->$attribute))
			$this->$attribute = null;
		else if (in_array($this->$attribute, array('en', 'ang', 'ang.', 'angielski', 'english', '1'))) {
			$this->czyEgzamin = 0;
			$this->czyPrzetarg = 0;
			$this->$attribute = '1';
		}
		else
			$this->addError($attribute, 'Nieprawidłowa wartość dla pola Język (' . $oldJ . ').');
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'dziedzina'=>array(self::BELONGS_TO, 'Dziedzina', 'idDziedziny'),
			'zlecenieFirmies'=>array(self::HAS_MANY, 'ZlecenieFirmy', 'idZlecenia'),
			'odpowiedziZlecenia'=>array(self::HAS_MANY, 'ZlecenieOdpowiedz', 'idZlecenia'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Zlecenie', 'ID'),
			'idDziedziny'=>Yii::t('Zlecenie', 'Dziedzina'),
			'poziom'=>Yii::t('Zlecenie', 'Poziom'),
			'kategoria'=>Yii::t('Zlecenie', 'Kategoria'),
			'czas'=>Yii::t('Zlecenie', 'Czas [gg:mm:ss]'),
			'cena'=>Yii::t('Zlecenie', 'Cena [zł]'),
			'kara'=>Yii::t('Zlecenie', 'Kara (% od ceny)'),
			'czyPrzetarg'=>Yii::t('Zlecenie', 'Przetarg'),
			'tresc'=>Yii::t('Zlecenie', 'Treść'),
			'wylaczone'=>Yii::t('Zlecenie', 'Wyłączone'),
			'urlObrazka'=>Yii::t('Zlecenie', 'Obrazek'),
			'moodle'=>Yii::t('Szkolenie', 'To zlecenie przekaż do platformy edukacyjnej'),
			'jednostka'=>Yii::t('Szkolenie', 'Jednostka'),
			'image'=>Yii::t('Szkolenie', 'Obrazek'),
			'czyEgzamin'=>Yii::t('Szkolenie', 'Pytanie egzaminacyjne'),
			'jezyk'=>Yii::t('Szkolenie', 'Język'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;
		$criteria->with = array('dziedzina');

		$criteria->compare('"t"."id"', $this->id, true);
		$criteria->compare('LOWER("dziedzina"."nazwa")', strtolower($this->idDziedziny), true);
		$criteria->compare('"poziom"', $this->getPoziomNum());
		$criteria->compare('LOWER("kategoria")', strtolower($this->kategoria), true);
		$criteria->compare('"czas"', $this->getCzasNum());
		$criteria->compare('"cena"', $this->cena);
		$criteria->compare('"kara"', $this->kara);
		$criteria->compare('"czyPrzetarg"', $this->czyPrzetarg, true);
		$criteria->compare('"czyEgzamin"', $this->czyEgzamin, true);
		$criteria->compare('LOWER("tresc")', strtolower($this->tresc), true);
		$criteria->compare('LOWER("urlObrazka")', strtolower($this->urlObrazka), true);
		$criteria->compare('LOWER("jednostka")', strtolower($this->jednostka), true);
		$criteria->compare('LOWER("jezyk")', strtolower($this->jezyk), true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
			'pagination'=>array(
				'pageSize'=>isset($_GET['pageSize']) ? $_GET['pageSize'] : 100,
			),
		));
	}

	static private $_poziomy = array(
		self::LEVEL_EASY => 'Łatwe',
		self::LEVEL_HARD => 'Trudne',
		self::LEVEL_VERY_HARD => 'Bardzo trudne',
	);

	static private $_poziomyEn = array(
		self::LEVEL_EASY => 'Easy',
		self::LEVEL_HARD => 'Hard',
		self::LEVEL_VERY_HARD => 'Very hard',
	);

	public function getPoziomStr()
	{
		return isset(self::$_poziomy[$this->poziom]) ? self::$_poziomy[$this->poziom] : '';
	}

	public function getPoziomStrEn()
	{
		return isset(self::$_poziomyEn[$this->poziom]) ? self::$_poziomyEn[$this->poziom] : '';
	}

	public function getPoziomNum()
	{
		if (empty($this->poziom))
			return '';
		if (is_numeric($this->poziom))
			return $this->poziom;
		$poziom = strtolower($this->poziom);
		foreach (self::$_poziomy as $k => $p)
			if (strpos(strtolower($p), $poziom) !== false)
				return $k;
		return -1;
	}

	public function getCzasNum()
	{
		if(is_numeric($this->czas))
			return $this->czas;
		else
			return $this->getParamDefaultValue('CZAS', $this->idDziedziny, $this->poziom);
	}

	public function getKaraNum()
	{
		$this->kara = (float)$this->kara;

		if($this->kara < 1)
			$this->kara *= 100;

		$this->kara = round($this->kara) % 100;

		if(empty($this->kara))
			$this->kara = $this->getParamDefaultValue('KARA', $this->idDziedziny, $this->poziom);

		return $this->kara;
	}

	public static function getCzasTrwaniaString($czas)
	{
		$result = '';
		$czas = (int)$czas;

		$hh = floor($czas / 3600) % 24;
		if ($hh) $result .= $hh . ' godz. ';

		$mm = floor($czas / 60) % 60;
		if ($mm) $result .= $mm . ' min. ';

		$ss = $czas % 60;
		if ((!$hh && !$mm) || $ss) $result .= $ss . ' sek. ';

		return $result;
	}

	public function getCzasTrwaniaStr()
	{
		return self::getCzasTrwaniaString($this->czas);
	}

	public static function getPoziomArray()
	{
		return self::$_poziomy;
	}

	public static function getPoziomString($level)
	{
		return isset(self::$_poziomy[$level]) ? self::$_poziomy[$level] : null;
	}

	public static function getPoziomStringEn($level)
	{
		return isset(self::$_poziomyEn[$level]) ? self::$_poziomyEn[$level] : null;
	}

	public function getKwotaKary()
	{
		return $this->kara / 100 * $this->cena;
	}

	public function getKwotaKaryCurr($curr = 'PLN')
	{
		return number_format($this->kwotaKary, 2, ',', ' ') . ' ' . $curr;
	}

	public function checkAnswer($answer)
	{
		$answer = strtolower(trim($answer));
		foreach ($this->odpowiedziZlecenia as $odpowiedz)
		{
			$odp = strtolower(trim($odpowiedz->odpowiedz));
			if ($answer == $odp)
				return $odpowiedz->id;
		}
		return null;
	}

	public function getScoreForOrder()
	{
		return 100 * pow(2, $this->poziom - 1);
	}

	public function canDoTask($companyId, $kontrahentId = null)
	{
		if ($kontrahentId)
		{
			//sprawdz czy mozna współpracować z kontrahentem
			if (!Kontrahent::companiesCanCooperate($companyId, $kontrahentId))
				throw new Exception('Firma odmówiła współpracy z powodu słabej reputacji.');
		}

		//sprawdza czy mozna podiąć się zlecenia
		$poziomy = Firma::getReputations($companyId);
		if ((int)$this->poziom > $poziomy[$this->idDziedziny])
			throw new Exception('Nie można podiąć się zlecenia z powodu braku doświadczenia lub słabej reputacji.');

		//sprawdz czy jest wystarczajaca ilosc ludzi
		//$leftTasks = Firma::getLeftTasksCount($companyId, Yii::app()->user->id_klasy);
		//if ($leftTasks <= 0)
		//	throw new Exception('Nie można podiąć się zlecenia z powodu baku siły roboczej.');

		return true;
	}

	private function getParamDefaultValue($param, $dziedzina, $poziom)
	{
		if(!in_array($dziedzina, array(1,2,3,4)) || !in_array($poziom, array(1,2,3)))
			return '';

		$dziedzinaToText = array(1=>'MAT', 2=>'FIZ', 3=>'BIOL', 4=>'CHEM');
		$poziomToText = array(1=>'LATWY', 2=>'SREDNI', 3=>'TRUDNY');

		if($param == 'CENA')
			return ParametryGry::getValue('DOMYSLNA_CENA_'.$dziedzinaToText[$dziedzina].'_'.$poziomToText[$poziom]);
		else if($param == 'KARA')
			return ParametryGry::getValue('DOMYSLNA_KARA_'.$dziedzinaToText[$dziedzina].'_'.$poziomToText[$poziom]);
		else if($param == 'CZAS')
			return ParametryGry::getValue('DOMYSLNY_CZAS_'.$dziedzinaToText[$dziedzina].'_'.$poziomToText[$poziom]);

		return '';
	}
}
